<?php

namespace App\Services;

use App\Models\Language;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Session;

class LanguageService
{
    /**
     * Resolve the current language for the application.
     * Priority: User preference -> Session -> System default -> 'en'
     */
    public function resolveLanguage(): string
    {
        $default_language = get_setting('system.active_language');

        return Auth::user() ? Session::get('locale', config('app.locale')) : (! empty($default_language) ? $default_language : 'en');
    }

    /**
     * Switch the application language.
     *
     * @param  string  $code  Language code
     * @param  bool  $persist  Whether to persist to user preferences
     */
    public function switchLanguage(string $code): void
    {
        $previousLanguage = $this->resolveLanguage();

        if (tenant_check()) {
            $tenant = current_tenant();
            $previousLanguage = $tenant->id.'_tenant_'.$previousLanguage;
        }

        // Clear translation cache for both old and new languages
        Cache::forget("translations.{$previousLanguage}");
        Session::put('locale', $code);
        App::setLocale($code);
    }

    /**
     * Get available languages for the current context.
     */
    public function getAvailableLanguages(): \Illuminate\Database\Eloquent\Collection
    {
        return Language::getCached();
    }

    /**
     * Check if a language code is valid and available.
     * Now accepts both active and inactive languages since they appear in the UI.
     */
    public function isValidLanguage(string $code): bool
    {
        return Language::getCached()->where('code', $code)->isNotEmpty();
    }
}
