<?php

namespace App\Livewire\Tenant\Settings\WhatsMark;

use App\Rules\PurifiedInput;
use Livewire\Component;

class WebHooksSettings extends Component
{
    public ?bool $enable_webhook_resend = false;

    public ?string $webhook_resend_method = '';

    public ?string $whatsapp_data_resend_to = '';

    public ?bool $only_agents_can_chat = false;

    protected function rules()
    {
        return [
            'enable_webhook_resend' => 'nullable|boolean',

            'webhook_resend_method' => [
                'nullable',
                'string',
                'max:255',
                new PurifiedInput(t('sql_injection_error')),
                'required_if:enable_webhook_resend,true',
            ],

            'whatsapp_data_resend_to' => [
                'nullable',
                'url',
                'max:255',
                new PurifiedInput(t('sql_injection_error')),
                'required_if:enable_webhook_resend,true',
            ],

            'only_agents_can_chat' => 'nullable|boolean',
        ];
    }

    public function mount()
    {
        if (! checkPermission('tenant.whatsmark_settings.view')) {
            $this->notify(['type' => 'danger', 'message' => t('access_denied_note')], true);

            return redirect(tenant_route('tenant.dashboard'));
        }

        $settings = tenant_settings_by_group('whats-mark');

        $this->enable_webhook_resend = $settings['enable_webhook_resend'] ?? false;
        $this->only_agents_can_chat = $settings['only_agents_can_chat'] ?? false;
        $this->webhook_resend_method = $settings['webhook_resend_method'] ?? null;
        $this->whatsapp_data_resend_to = $settings['whatsapp_data_resend_to'] ?? null;
    }

    public function save()
    {
        if (checkPermission('tenant.whatsmark_settings.edit')) {
            $this->validate();

            $settings = tenant_settings_by_group('whats-mark');

            $originalSettings = [
                'enable_webhook_resend' => $settings['enable_webhook_resend'] ?? false,
                'only_agents_can_chat' => $settings['only_agents_can_chat'] ?? false,
                'webhook_resend_method' => $settings['webhook_resend_method'] ?? null,
                'whatsapp_data_resend_to' => $settings['whatsapp_data_resend_to'] ?? null,
            ];

            $modifiedSettings = [];

            foreach ($originalSettings as $key => $originalValue) {
                $newValue = $this->{$key};

                if (! array_key_exists($key, $settings) || $originalValue !== $newValue) {
                    $modifiedSettings[$key] = $newValue;
                }
            }

            if (! empty($modifiedSettings)) {
                foreach ($modifiedSettings as $key => $value) {
                    save_tenant_setting('whats-mark', $key, $value);
                }

                $this->notify([
                    'type' => 'success',
                    'message' => t('setting_save_successfully'),
                ]);
            }
        }
    }

    public function render()
    {
        return view('livewire.tenant.settings.whats-mark.web-hooks-settings');
    }
}
