<?php

namespace App\Livewire\Admin\Tables;

use App\Enum\TenantStatus;
use App\Events\Tenant\TenantStatusChanged;
use App\Models\Tenant;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use PowerComponents\LivewirePowerGrid\Button;
use PowerComponents\LivewirePowerGrid\Column;
use PowerComponents\LivewirePowerGrid\Facades\PowerGrid;
use PowerComponents\LivewirePowerGrid\PowerGridComponent;
use PowerComponents\LivewirePowerGrid\PowerGridFields;

final class TenantTable extends PowerGridComponent
{
    public string $tableName = 'tenant-table';

    public string $sortField = 'created_at';

    public string $sortDirection = 'DESC';

    public bool $deferLoading = true;

    public string $loadingComponent = 'components.custom-loading';

    public function setUp(): array
    {
        return [
            PowerGrid::header()
                ->showSearchInput()
                ->showToggleColumns()
                ->withoutLoading(),
            PowerGrid::footer()
                ->showPerPage()
                ->showRecordCount(),
        ];
    }

    public function datasource(): Builder
    {
        return Tenant::query()
            ->join('users', 'tenants.id', '=', 'users.tenant_id')
            ->join(DB::raw('(
                    SELECT MIN(id) as id
                    FROM users
                    WHERE is_admin = 1
                    GROUP BY tenant_id
                ) as oldest_admins'), 'users.id', '=', 'oldest_admins.id')
            ->select([
                'tenants.id',
                'tenants.company_name',
                'tenants.status',
                'tenants.domain',
                'tenants.created_at',
                'users.id as user_id',
                'users.firstname',
                'users.lastname',
                'users.email',
            ]);
    }

    public function fields(): PowerGridFields
    {
        $options = $this->statusSelectOptions();

        return PowerGrid::fields()
            ->add('firstname', function ($user) {
                $loggedInUser = Auth::user();

                // Start rendering output
                $output = '<div class="group relative inline-block min-h-[40px]">
                <div class="flex items-center gap-3 w-auto min-w-0 max-w-full ">
                    <p class="dark:text-gray-200 text-indigo-600 dark:hover:text-indigo-400 text-sm break-words truncate">'.
                    $user->firstname.' '.$user->lastname.
                    '</p>
                </div>

                <!-- Action Links (Hidden by Default, Shown on Hover) -->
                 <div class="absolute contact-actions left-[-40px] lg:left-0 top-3 mt-2  hidden  space-x-1 text-xs text-gray-600 dark:text-gray-300">';

                $hasPrevious = false;

                if (checkPermission('admin.tenants.view')) {
                    $output .= '<button onclick="Livewire.dispatch(\'viewTenant\', { tenantId: '.$user->id.' })" class="hover:text-blue-600">'.t('view').'</button>';
                    $hasPrevious = true;
                }

                if (checkPermission('admin.tenants.edit')) {
                    if ($hasPrevious) {
                        $output .= ' <span class="pt-3 lg:pt-1">|</span>';
                    }
                    $output .= '<button onclick="Livewire.dispatch(\'editTenant\', { tenantId: '.$user->id.' })" class="hover:text-green-600">'.t('edit').'</button>';
                    $hasPrevious = true;
                }

                if (checkPermission('admin.tenants.delete')) {
                    if ($hasPrevious) {
                        $output .= ' <span class="pt-3 lg:pt-1">|</span>';
                    }
                    $output .= '<button onclick="Livewire.dispatch(\'confirmDelete\', { tenantId: '.$user->id.' })" class="hover:text-red-600">'.t('delete').'</button>';
                }

                $output .= '</div></div>';

                return $output;
            })

            ->add('created_at_formatted', function ($user) {
                return '<div class="relative group">
                        <span class="cursor-default" data-tippy-content="'.format_date_time($user->created_at).'">'
                    .Carbon::parse($user->created_at)->diffForHumans(['options' => Carbon::JUST_NOW]).'</span>
                    </div>';
            })
            ->add('status', function ($user) use ($options) {
                return Blade::render(
                    '<x-select-status type="occurrence" :options="$options" :userId="$userId" :selected="$selected"/>',
                    [
                        'options' => $options,
                        'userId' => intval($user->id),
                        'selected' => $user->status,
                    ]
                );
            });
    }

    #[On('statusChanged')]
    public function statusChanged($statusId, $userId)
    {
        if (checkPermission('admin.tenants.edit')) {
            // Find the tenant by ID
            $tenant = Tenant::find($userId);

            if ($tenant) {
                // Store original status for comparison
                $originalStatus = $tenant->status;

                // Update tenant status directly with the string value
                $tenant->status = $statusId;

                // Save the tenant
                $tenant->save();

                Cache::forget("tenant_{$tenant->id}");

                event(new TenantStatusChanged($tenant, $originalStatus, $statusId));

                // Show success notification
                $this->notify([
                    'message' => t('tenant_status_updated', ['status' => ucfirst($statusId)]),
                    'type' => 'success',
                ]);
            } else {
                // Tenant not found - show error
                $this->notify([
                    'message' => t('tenant_not_found'),
                    'type' => 'error',
                ]);
            }
        } else {
            $this->notify(['type' => 'danger', 'message' => t('access_denied_note')], true);

            return redirect(route('admin.tenants.list'));
        }
    }

    public function columns(): array
    {
        return [
            Column::make('ID', 'id')
                ->searchable()
                ->sortable(),

            Column::make('Name', 'firstname', 'users.firstname')
                ->bodyAttribute('class="relative"')
                ->searchable()
                ->sortable(),
            Column::make('Company Name', 'company_name')
                ->searchable()
                ->sortable(),
            Column::make('Status', 'status')
                ->searchable()
                ->sortable(),
            Column::make('Created at', 'created_at_formatted', 'created_at')
                ->searchable(),

            Column::action(t('action'))
                ->hidden(! checkPermission(['admin.tenants.login'])),
        ];
    }

    public function filters(): array
    {
        return [];
    }

    public function statusSelectOptions()
    {
        $labels = TenantStatus::labels();

        if (empty($labels)) {
            // Fallback in case the enum method fails
            return [
                'active' => 'Active',
                'deactive' => 'Deactive',
                'suspended' => 'Suspended',
                'expired' => 'Expired',
            ];
        }

        return $labels;
    }

    public function actions(Tenant $tenant)
    {
        $actions = [];
        if (checkPermission('admin.tenants.login')) {
            $actions[] = Button::add('login_as_tenant')
                ->slot(t('login_as_tenant'))
                ->id()
                ->class('inline-flex items-center justify-center px-3 py-1 text-sm border border-green-300 rounded-md font-medium disabled:opacity-50 disabled:pointer-events-none transition bg-green-100 text-green-700 hover:bg-green-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-300 dark:bg-slate-700 dark:border-slate-500 dark:text-green-400 dark:hover:border-green-600 dark:hover:bg-green-600 dark:hover:text-white dark:focus:ring-offset-slate-800')
                ->route('admin.login.as', ['id' => $tenant->user_id]);
        }

        return $actions ?? [];
    }
}
